#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kobject.h>
#include <linux/sysfs.h>
#include <linux/netdevice.h>
#include <linux/vmalloc.h>
#include <dvl_features.h>

#include "dvltrafin_mod.h"
#include "athrs_phy_ctrl.h"

//####################################
//external prototypes
extern unsigned int athrs27_reg_read(unsigned int); //in module athrs_gmac included in package.ar934x-eth
extern void athrs27_reg_write(unsigned int, unsigned int); //in module athrs_gmac included in package.ar934x-eth
extern int athrs27_phy_is_link_alive(int); //in module athrs_gmac included in package.ar934x-eth


//####################################
//external variables
extern int devolo_athrs_mac_open; //in modules athrs_gmac included in package.ar934x-eth


//####################################
//local variables
#ifndef _DVL_CLASSIC_LED_BLINK_FREQ
    //these values have to be determined empirically
	static const unsigned int cTrafficIntensityLow = 5;
	static const unsigned int cTrafficIntensityMedium = 20;
	static const unsigned int cTrafficIntensityHigh = 60;
#endif


//####################################
//sysfs entry

#define NUM_PORTS 5

struct stats_kobj_attribute
{
    struct kobj_attribute kattr;
    unsigned int portnum;
};

struct stats_entry
{
    const char* name;
    unsigned int reg;
    BOOL led;
    unsigned long long int value[NUM_PORTS];
};

#define INIT_STATS_ENTRY(_name, _reg, _led)  { .name = __stringify(_name), .reg = _reg, .led = _led },

static struct stats_entry stats_entries[] =
{
    INIT_STATS_ENTRY(all, 0xffffffff, FALSE)  // fake entry for summary, must be the first element
    
    INIT_STATS_ENTRY(rx_broad,        ATHR_PHY_RX_BROAD_REG,       FALSE)
    INIT_STATS_ENTRY(rx_pause,        ATHR_PHY_RX_PAUSE_REG,       FALSE)
    INIT_STATS_ENTRY(rx_multi,        ATHR_PHY_RX_MULTI_REG,       FALSE)
    INIT_STATS_ENTRY(rx_fcserr,       ATHR_PHY_RX_FCSERR_REG,      FALSE)
    INIT_STATS_ENTRY(rx_allignerr,    ATHR_PHY_RX_ALIGNERR_REG,    FALSE)
    INIT_STATS_ENTRY(rx_runt,         ATHR_PHY_RX_RUNT_REG,        FALSE)
    INIT_STATS_ENTRY(rx_frag,         ATHR_PHY_RX_FRAGMENT_REG,    FALSE)
    INIT_STATS_ENTRY(rx_64b,          ATHR_PHY_RX_64B_REG,         TRUE )
    INIT_STATS_ENTRY(rx_128b,         ATHR_PHY_RX_128B_REG,        TRUE )
    INIT_STATS_ENTRY(rx_256b,         ATHR_PHY_RX_512B_REG,        TRUE )
    INIT_STATS_ENTRY(rx_512b,         ATHR_PHY_RX_1024B_REG,       TRUE )
    INIT_STATS_ENTRY(rx_1024b,        ATHR_PHY_RX_1518B_REG,       TRUE )
    INIT_STATS_ENTRY(rx_1518b,        ATHR_PHY_RX_MAXB_REG,        TRUE )
    INIT_STATS_ENTRY(rx_maxb,         ATHR_PHY_RX_TOLO_REG,        FALSE)
    INIT_STATS_ENTRY(rx_tool,         ATHR_PHY_RX_GOODBL_REG,      FALSE)
    INIT_STATS_ENTRY(rx_goodbl,       ATHR_PHY_RX_GOODBU_REG,      FALSE)
    INIT_STATS_ENTRY(rx_goodbh,       ATHR_PHY_RX_BADBL_REG,       FALSE)
    INIT_STATS_ENTRY(rx_overflow,     ATHR_PHY_RX_BADBU_REG,       FALSE)
    INIT_STATS_ENTRY(rx_badbl,        ATHR_PHY_RX_OVERFLW_REG,     FALSE)
    INIT_STATS_ENTRY(rx_badbu,        ATHR_PHY_RX_FILTERD_REG,     FALSE)
    INIT_STATS_ENTRY(tx_broad,        ATHR_PHY_TX_BROAD_REG,       FALSE)
    INIT_STATS_ENTRY(tx_pause,        ATHR_PHY_TX_PAUSE_REG,       FALSE)
    INIT_STATS_ENTRY(tx_multi,        ATHR_PHY_TX_MULTI_REG,       FALSE)
    INIT_STATS_ENTRY(tx_underrun,     ATHR_PHY_TX_UNDERRN_REG,     FALSE)
    INIT_STATS_ENTRY(tx_64b,          ATHR_PHY_TX_64B_REG,         TRUE )
    INIT_STATS_ENTRY(tx_128b,         ATHR_PHY_TX_128B_REG,        TRUE )
    INIT_STATS_ENTRY(tx_256b,         ATHR_PHY_TX_256B_REG,        TRUE )
    INIT_STATS_ENTRY(tx_512b,         ATHR_PHY_TX_512B_REG,        TRUE )
    INIT_STATS_ENTRY(tx_1024b,        ATHR_PHY_TX_1024B_REG,       TRUE )
    INIT_STATS_ENTRY(tx_1518b,        ATHR_PHY_TX_1518B_REG,       TRUE )
    INIT_STATS_ENTRY(tx_maxb,         ATHR_PHY_TX_MAXB_REG,        TRUE )
    INIT_STATS_ENTRY(tx_oversiz,      ATHR_PHY_TX_OVSIZE_REG,      FALSE)
    INIT_STATS_ENTRY(tx_bytel,        ATHR_PHY_TX_TXBYTEL_REG,     FALSE)
    INIT_STATS_ENTRY(tx_byteh,        ATHR_PHY_TX_TXBYTEU_REG,     FALSE)
    INIT_STATS_ENTRY(tx_collision,    ATHR_PHY_TX_COLL_REG,        FALSE)
    INIT_STATS_ENTRY(tx_abortcol,     ATHR_PHY_TX_ABTCOLL_REG,     FALSE)
    INIT_STATS_ENTRY(tx_multicol,     ATHR_PHY_TX_MLTCOL_REG,      FALSE)
    INIT_STATS_ENTRY(tx_singalcol,    ATHR_PHY_TX_SINGCOL_REG,     FALSE)
    INIT_STATS_ENTRY(tx_execdefer,    ATHR_PHY_TX_EXDF_REG,        FALSE)
    INIT_STATS_ENTRY(tx_defer,        ATHR_PHY_TX_DEF_REG,         FALSE)
    INIT_STATS_ENTRY(tx_latecol,      ATHR_PHY_TX_LATECOL_REG,     FALSE)
};
const size_t num_stats_entries = (sizeof(stats_entries) / sizeof(struct stats_entry));

static void stats_entries_clear(void)
{
    size_t i, j;
    for (i = 0; i < num_stats_entries; i++)
    {
        for (j = 0; j < NUM_PORTS; j++)
        {
            stats_entries[i].value[j] = 0ULL;
        }
    }
}

static unsigned long long int stats_entry_get_value(const char* name, unsigned int portnum)
{
    if (portnum < NUM_PORTS)
    {
        size_t i;
        for (i = 0; i < num_stats_entries; i++)
        {
            if (!strcmp(name, stats_entries[i].name))
            {
                return stats_entries[i].value[portnum];
            }
        }
    }
    return 0;
}

static unsigned int stats_read_and_add(unsigned int portnum)
{
    size_t i;
    unsigned int packetcount = 0;
    
    unsigned int addr = 0x20000;  // base address for mib counters
    addr |= (0x100 * portnum);    // set port (starting at port (i.e. mac) no. 1!)
    
    for (i = 0; i < num_stats_entries; i++)
    {
        if (stats_entries[i].reg != 0xffffffff)
        {
            unsigned int value = athrs27_reg_read(addr | stats_entries[i].reg);
            
            if (stats_entries[i].led)
            {
                packetcount += value;
            }
            
            if (portnum < NUM_PORTS)
            {
                // since we use 64-bit counters, we don't need special handling for 2x32-bit values (e.g. byte counters)
                stats_entries[i].value[portnum] += value;
            }
        }
    }
    
    return packetcount;
}

#define LINE_SIZE 64
static ssize_t stats_show(struct kobject *kobj, struct kobj_attribute *attr, char *buf)
{
    unsigned int portnum = ((struct stats_kobj_attribute*) attr)->portnum;
    if (strcmp(attr->attr.name, "all") != 0)
    {
        return snprintf(buf, PAGE_SIZE, "%llu", stats_entry_get_value(attr->attr.name, portnum));
    }
    else
    {
        *buf = 0;
        if (portnum < NUM_PORTS)
        {
            if (PAGE_SIZE >= LINE_SIZE * num_stats_entries)
            {
                size_t i;
                for (i = 1; i < num_stats_entries; i++)
                {
                    char line[LINE_SIZE];
                    snprintf(line, sizeof(line), "%-32s=%llu\n", stats_entries[i].name, stats_entries[i].value[portnum]);
                    strcat(buf, line);
                }
            }
            else
            {
                printk(KERN_ERR "Summary does not fit into PAGE_SIZE\n");
            }
        }
        return strlen(buf);
    }
}

static ssize_t stats_store(struct kobject *kobj, struct kobj_attribute *attr, const char *buf, size_t len)
{
    return 0;
}

static struct stats_kobj_attribute *(stats_kobj_attrs[NUM_PORTS]);
static struct attribute **(stats_attrs[NUM_PORTS]);
static struct attribute_group stats_group[NUM_PORTS];

static struct kobject *stats_kobject;
static struct kobject *(stats_kobject_port[NUM_PORTS]);

static void stats_destroy(void)
{
    size_t i;
    
    for (i = 0; i < NUM_PORTS; i++)
    {
        if (stats_kobject_port[i])
        {
            kobject_put(stats_kobject_port[i]);
            stats_kobject_port[i] = NULL;
        }
    }
    
    if (stats_kobject)
    {
        kobject_put(stats_kobject);
        stats_kobject = NULL;
    }

    for (i = 0; i < NUM_PORTS; i++)
    {
        if (stats_kobj_attrs[i])
        {
            vfree(stats_kobj_attrs[i]);
            stats_kobj_attrs[i] = NULL;
        }
        if (stats_attrs[i])
        {
            vfree(stats_attrs[i]);
            stats_attrs[i] = NULL;
        }
    }
}

static BOOL stats_init(void)
{
    size_t i, j;

    stats_entries_clear();
    
    stats_kobject = NULL;
    for (i = 0; i < NUM_PORTS; i++)
    {
        stats_kobject_port[i] = NULL;
        stats_kobj_attrs[i] = NULL;
        stats_attrs[i] = NULL;
    }
    
    stats_kobject = kobject_create_and_add("dvltrafin_eth", kernel_kobj);
    if (!stats_kobject)
    {
        stats_destroy();
        return FALSE;
    }
    
    for (i = 0; i < NUM_PORTS; i++)
    {
        char name[128];
        snprintf(name, sizeof(name), "port%d", i);
        
        stats_kobject_port[i] = kobject_create_and_add(name, stats_kobject);
        if (!stats_kobject_port[i])
        {
            stats_destroy();
            return FALSE;
        }
    }

    for (i = 0; i < NUM_PORTS; i++)
    {
        stats_kobj_attrs[i] = vmalloc(sizeof(struct stats_kobj_attribute) * num_stats_entries);
        stats_attrs[i] = vmalloc(sizeof(void*) * (num_stats_entries + 1));
        if (!stats_kobj_attrs[i] || !stats_attrs[i])
        {
            stats_destroy();
            return FALSE;
        }
    }
    
    for (i = 0; i < NUM_PORTS; i++)
    {
        for (j = 0; j < num_stats_entries; j++)
        {
            stats_kobj_attrs[i][j].portnum = i;
            
            stats_kobj_attrs[i][j].kattr.attr.name = stats_entries[j].name;
            stats_kobj_attrs[i][j].kattr.attr.mode = 0444;
            stats_kobj_attrs[i][j].kattr.show  = stats_show;
            stats_kobj_attrs[i][j].kattr.store = stats_store;
            
            stats_attrs[i][j] = &(stats_kobj_attrs[i][j].kattr.attr);
        }
        stats_attrs[i][num_stats_entries] = NULL;
    }
    
    for (i = 0; i < NUM_PORTS; i++)
    {
        stats_group[i].attrs = stats_attrs[i];

        if (sysfs_create_group(stats_kobject_port[i], &stats_group[i]) != 0)
        {
            stats_destroy();
            return FALSE;
        }
    }
    
    return TRUE;
}

//####################################
//local prototypes
static void enableMibCounter(void);
static void disableMibCounter(void);

//####################################
//local function definitions

static BOOL isMibCounterEnabled(void)
{
	return ((athrs27_reg_read(ATHR_PHY_MIB_CTRL_REG) & ATHR_PHY_MIB_ENABLE) == ATHR_PHY_MIB_ENABLE) ? TRUE : FALSE;
}

static void enableMibCounter(void)
{
	athrs27_reg_write(ATHR_PHY_MIB_CTRL_REG, (athrs27_reg_read(ATHR_PHY_MIB_CTRL_REG) | ATHR_PHY_MIB_ENABLE));
}

static void disableMibCounter(void)
{
	athrs27_reg_write(ATHR_PHY_MIB_CTRL_REG, (athrs27_reg_read(ATHR_PHY_MIB_CTRL_REG) & ~ATHR_PHY_MIB_ENABLE));
}

BOOL connection_active(void)
{
	if (athrs27_phy_is_link_alive(1) || athrs27_phy_is_link_alive(2) || athrs27_phy_is_link_alive(3))
	{
		return TRUE;
	}
	else
	{
		return FALSE;
	}
}

traffic_intensity_t get_traffic_intensity()
{
	unsigned int rv = eTRAFFIC_NONE;
	unsigned int packetSum = 0;

	if (!devolo_athrs_mac_open)
	{
	    return rv;
	}

	if (!isMibCounterEnabled())
	{
		stats_entries_clear();
		enableMibCounter();
	}

	//check each port separately and sum up packet counter over all ports
	//phy ports start at 0, mac ports (used for accessing mib counters) at 1!
	//ADDITIONALLY (!), mounted ports are phy1, phy2 and phy3
	if (athrs27_phy_is_link_alive(1))
	{
		packetSum += stats_read_and_add(2);
	}

	if (athrs27_phy_is_link_alive(2))
	{
		packetSum += stats_read_and_add(3);
	}

	if (athrs27_phy_is_link_alive(3))
	{
		packetSum += stats_read_and_add(4);
	}

	//return accordant traffic intensity category
#ifndef _DVL_CLASSIC_LED_BLINK_FREQ

	if (packetSum > 0 && packetSum <= cTrafficIntensityLow)
	{
		rv = eTRAFFIC_LOW;
	}
	else if (packetSum > cTrafficIntensityLow && packetSum <= cTrafficIntensityMedium)
	{
		rv = eTRAFFIC_MEDIUM;
	}
	else if (packetSum > cTrafficIntensityHigh)
	{
		rv = eTRAFFIC_HIGH;
	}
	else //packetSum == 0
	{
		rv = eTRAFFIC_NONE;
	}

#else
	if (packetSum > 0)
	{
		rv = eTRAFFIC_HIGH;
	}
	else
	{
		rv = eTRAFFIC_NONE;
	}

#endif

	return rv;
}

BOOL switchctrl_load(char *devname)
{
    if (devname != 0 && strcmp(devname, "eth1") != 0)
    {
        printk(KERN_ERR "Setting a device is not supported by this module variant.\n");
    }

    if (!stats_init())
    {
        printk(KERN_ERR "Unable to create kernel objects and attribute groups\n");
        return FALSE;
    }
    
    return TRUE;
}

BOOL switchctrl_unload(void)
{
	stats_destroy();
	disableMibCounter();
	return TRUE;
}

